/* ----------------------------------------------------------------------
* Copyright (C) 2010 ARM Limited. All rights reserved.  
* 
* $Date: 		26. Aug 2010  		 
* $Revision: 	V0.0.9  
* 
* Project:      Zephyr    
* Title:	    arm_linear_interp_f32.c		 
* Author(s):	KK      
* Revised by:		 
* 
* Description:	Example code demonstrating usage of sin function 
*               and uses linear interpolation to get higher precision
*				
*				
* References:
*
* Target Processor:          Cortex-M4
* Target Tools Revision:     MDK uVision V4.00o
* ------------------------------------------------------------------- */

/**
 * @ingroup groupExamples
 */

/**   
 * @defgroup LinearInterpExample Linear Interpolate Example   
 *
 * <b> Zephyr Library Linear Interpolate Example  </b>
 *
 * First Calculate sine values using cubic interpolation and then linear interpoltion 
 * to get higher precision 
 *
 * <b> Zephyr Functions Used </b>
 * arm_sin_f32, arm_linear_interp_f32
 *
 * <b> Description </b>
 * First Calculate sine values using cubic interpolation and then linear interpoltion 
 * to get higher precision 
 *
 * <b> Refer  </b>
 * \link arm_linear_interp_f32.c
 *
 */


/** \example arm_linear_interp_f32.c
  */ 

 /** \link */
    
#include "arm_math.h"
#include "math_helper.h"

#define SNR_THRESHOLD 			90
#define TEST_LENGTH_SAMPLES 	100

/* ----------------------------------------------------------------------
* Test input data for sin_cos F32, Q31 Generated by the 
* MATLAB rand() function in the range of -180 to 179 degrees
* --------------------------------------------------------------------*/
float32_t testInput_f32[TEST_LENGTH_SAMPLES] = 
{   
	-19.465416518769931000,	-74.951297020714364000,	5.639953791367380700,	12.945438916134698000,	-51.591210780665868000,	53.591195953934943000,	53.512389074344640000,	-1.693497446699294000,	
14.728156263389437000,	7.858761426941603300,	-8.401885995664772000,	32.660574673198624000,	-26.474244435638489000,	98.243361818869545000,	-6.137814738896806500,	5.126909108436432700,	
48.004569511163496000,	2.667665723562240700,	-4.304178246765106400,	-37.455725864251015000,	13.248486737668816000,	-60.128183607201180000,	32.144604831852845000,	73.060292900082189000,	
-31.129906576602902000,	38.609850277271818000,	56.430063972113956000,	-71.717830940136452000,	-64.843399435545905000,	25.701643064618008000,	-17.994850997191342000,	31.049881895895531000,	
36.703002999426445000,	32.035874557540197000,	58.061238971961465000,	30.087022755691816000,	53.587713340951609000,	-54.110570164827479000,	-0.890530099594670220,	-7.052278447439130500,	
-72.183850290052135000,	11.578690560487043000,	-47.541281763666710000,	63.681366864255239000,	-36.229068188859593000,	23.793435493300120000,	9.869430270043006100,	-41.485573095999264000,	
-97.680352243736806000,	-2.663452103453602900,	-45.478516791341136000,	27.650837200296642000,	22.848335340389347000,	76.159344158573461000,	26.607716411587916000,	-28.961784120713677000,	
17.115176327125955000,	-45.410198595335324000,	-0.877980128863018240,	-2.169935511539052200,	0.001943632873149526,	-14.303675306145955000,	49.275168245437158000,	-84.329561593843238000,	
19.268247287032327000,	40.303731204528830000,	32.893080229325399000,	26.003580584885931000,	1.814131422829813100,	30.469013441878712000,	25.600509234032536000,	-11.504043703438416000,	
-16.986102998506258000,	-13.314919950160068000,	-66.381052763486679000,	-10.530182144521486000,	5.330017667435458700,	14.166406952777512000,	64.957870995741928000,	-15.793863224748380000,	
28.045523301232237000,	35.957187816650020000,	42.340047332750117000,	-44.644128099470784000,	9.541581847427494300,	10.704693279401042000,	-45.349352625522059000,	-33.392013846012176000,	
48.703272891990011000,	-5.917486632537308100,	17.544622035916753000,	3.959419796090685800,	-28.595935146569229000,	-25.180798598830847000,	19.964407027665033000,	-42.745670934644046000,	
35.153172804527614000,	25.603229057547324000,	-36.977143126331505000,	-11.952308309964708000
};

/*------------------------------------------------------------------------------
*  Reference out of F32 sin & cos output for SIN_COS Module with Block Size = 300
*------------------------------------------------------------------------------*/
float32_t testRefOutput32_f32[2*TEST_LENGTH_SAMPLES] =  
{ 
	-0.333237824190935950,	0.942842803721007820,	-0.965705474086925530,	0.259640014863938890,	0.098276873131171796,	0.995159111000627710,	0.224023088564911880,	0.974583837229941800,	
-0.783598163528961720,	0.621267992185368100,	0.804802603350386110,	0.593542559249495730,	0.803985460437475980,	0.594648954766709090,	-0.029552802881316703,	0.999563220532777570,	
0.254233248161875150,	0.967142934384086940,	0.136731592422420860,	0.990608132226678760,	-0.146115592196457200,	0.989267523836236400,	0.539661147574512780,	0.841882323011096330,	
-0.445795477686068590,	0.895134845749315010,	0.989668004962534040,	-0.143377961882144740,	-0.106920302934515420,	0.994267594171906800,	0.089362080262683524,	0.995999206129766780,	
0.743198188297227610,	0.669071336190483200,	0.046542730140767506,	0.998916299932603780,	-0.075051445518127535,	0.997179663112740980,	-0.608148199836544220,	0.793823511264040830,	
0.229174682523673430,	0.973385311626476900,	-0.867141848541994500,	0.498061255778014410,	0.532057903834495470,	0.846707970298640090,	0.956611891979700090,	0.291365214332491460,	
-0.516980202695697330,	0.855997353979973230,	0.624013946309263550,	0.781413203632712940,	0.833211499046846440,	0.552954426563443850,	-0.949523148821079350,	0.313696971379869070,	
-0.905149304940980780,	0.425093796431869010,	0.433684924493513650,	0.901064584958844890,	-0.308931524481638130,	0.951084282901074500,	0.515784132327432480,	0.856718582055529860,	
0.597667169002062740,	0.801744320277393510,	0.530450147955215390,	0.847716132047922070,	0.848614000856354080,	0.529012549520870850,	0.501314771271737620,	0.865264988373368400,	
0.804766524528853890,	0.593591476520468290,	-0.810149803083192290,	0.586222906891452730,	-0.015542056546419188,	0.999879214944639160,	-0.122774922462590140,	0.992434541123143780,	
-0.952043189926617960,	0.305963665349906650,	0.200713583736860770,	0.979649966724598140,	-0.737763909965450160,	0.675058821994417800,	0.896342291988333040,	0.443362713354542080,	
-0.591014991291254010,	0.806660573022506710,	0.403440464710986050,	0.915005896939349530,	0.171403478200279900,	0.985200917407635270,	-0.662431442426691610,	0.749122542768866920,	
-0.991029087727446800,	-0.133646351533083770,	-0.046469268167626407,	0.998919720055603480,	-0.712987591183941730,	0.701176650222838260,	0.464082160284159160,	0.885792158751694920,	
0.388293142209411300,	0.921535911244451670,	0.970964777276470790,	0.239222493274469370,	0.447879505598888560,	0.894093925974500330,	-0.484226147296217320,	0.874942877149509690,	
0.294293482382983140,	0.955715096787163270,	-0.712151017299339720,	0.702026301900089410,	-0.015323044321591962,	0.999882595264423250,	-0.037863466280923866,	0.999282921860067290,	
0.000033922793079950,	0.999999999424622030,	-0.247061170884917000,	0.968999885366852780,	0.757851648246485540,	0.652426914872528130,	-0.995106681257627580,	0.098806340466745052,	
0.329991297513742620,	0.943983974209942070,	0.646839444476510090,	0.762626207961226490,	0.543073042762871940,	0.839685459100177440,	0.438427314265245950,	0.898766649418058510,	
0.031657276274742796,	0.999498782820101630,	0.507072304663452660,	0.861903520031850560,	0.432093764101541660,	0.901828686073226080,	-0.199437093026770570,	0.979910631601183970,	
-0.292139745905651280,	0.956375642131365390,	-0.230303147358009550,	0.973118934312242720,	-0.916230287372644450,	0.400652044174295340,	-0.182753457037788390,	0.983158773515619530,	
0.092892242015230150,	0.995676167924784640,	0.244738948943198060,	0.969589009256075830,	0.905996795154595410,	0.423284546339223300,	-0.272177185260248640,	0.962247151112336830,	
0.470172944197104090,	0.882574304262834790,	0.587180579792218600,	0.809455969596168940,	0.673529320002908660,	0.739160506991830220,	-0.702701233357562740,	0.711485050185708510,	
0.165763349389780660,	0.986165560136370580,	0.185747103699765100,	0.982597584704516080,	-0.711405090256715830,	0.702782183579545580,	-0.550364369734323230,	0.834924583736124660,	
0.751301833382328810,	0.659958752617427980,	-0.103096114611431640,	0.994671398579463850,	0.301448464375417170,	0.953482471430756260,	0.069049923419193845,	0.997613205644253380,	
-0.478629568167083070,	0.878016934048650110,	-0.425476035077883510,	0.904969692075046070,	0.341436326633237750,	0.939904907347121510,	-0.678745260797874890,	0.734373795110109250,	
0.575764280402292440,	0.817615737013929090,	0.432136573310053560,	0.901808173620002540,	-0.601496377444140970,	0.798875527176527990,	-0.207097431303776860,	0.978320322771318860,	

}; 


float32_t arm_linear_interep_Sintable[720] =  {

  -0.008726535498374403,	-0.017452406437283439,	-0.026176948307873423,	-0.034899496702500699,	-0.043619387365336069,	-0.052335956242943807,	-0.061048539534857192,	-0.069756473744125524,	
-0.078459095727845068,	-0.087155742747658638,	-0.095845752520223912,	-0.104528463267653730,	-0.113203213767906890,	-0.121869343405147550,	-0.130526192220051570,	-0.139173100960065740,	
-0.147809411129610380,	-0.156434465040230980,	-0.165047605860678100,	-0.173648177666930280,	-0.182235525492147720,	-0.190808995376544970,	-0.199367934417197690,	-0.207911690817759310,	
-0.216439613938103180,	-0.224951054343864780,	-0.233445363855905530,	-0.241921895599667730,	-0.250380004054441330,	-0.258819045102521020,	-0.267238376078257010,	-0.275637355816999660,	
-0.284015344703922590,	-0.292371704722737050,	-0.300705799504273340,	-0.309016994374947510,	-0.317304656405092140,	-0.325568154457156980,	-0.333806859233771180,	-0.342020143325668880,	
-0.350207381259467540,	-0.358367949545300210,	-0.366501226724297140,	-0.374606593415912240,	-0.382683432365089890,	-0.390731128489274160,	-0.398749068925246530,	-0.406736643075800430,	
-0.414693242656239190,	-0.422618261740699500,	-0.430511096808295080,	-0.438371146789077290,	-0.446197813109808990,	-0.453990499739546860,	-0.461748613235033910,	-0.469471562785891080,	
-0.477158760259608630,	-0.484809620246337170,	-0.492423560103467160,	-0.499999999999999940,	-0.507538362960704430,	-0.515038074910054380,	-0.522498564715948910,	-0.529919264233204900,	
-0.537299608346823780,	-0.544639035015026860,	-0.551936985312058260,	-0.559192903470746900,	-0.566406236924833160,	-0.573576436351046380,	-0.580702955710939880,	-0.587785252292473250,	
-0.594822786751341260,	-0.601815023152048160,	-0.608761429008720430,	-0.615661475325658400,	-0.622514636637619630,	-0.629320391049837720,	-0.636078220277764170,	-0.642787609686539470,	
-0.649448048330183770,	-0.656059028990507280,	-0.662620048215737390,	-0.669130606358858350,	-0.675590207615660350,	-0.681998360062498590,	-0.688354575693753910,	-0.694658370458997140,	
-0.700909264299851010,	-0.707106781186547570,	-0.713250449154181560,	-0.719339800338651410,	-0.725374371012287740,	-0.731353701619170570,	-0.737277336810124080,	-0.743144825477394240,	
-0.748955720789002080,	-0.754709580222771790,	-0.760405965600031040,	-0.766044443118978010,	-0.771624583387720220,	-0.777145961456971010,	-0.782608156852414030,	-0.788010753606722010,	
-0.793353340291235170,	-0.798635510047292720,	-0.803856860617217390,	-0.809016994374947450,	-0.814115518356319230,	-0.819152044288992020,	-0.824126188622015810,	-0.829037572555041740,	
-0.833885822067168210,	-0.838670567945424050,	-0.843391445812885830,	-0.848048096156426070,	-0.852640164354092290,	-0.857167300702112330,	-0.861629160441525710,	-0.866025403784438710,	
-0.870355695939899700,	-0.874619707139395850,	-0.878817112661965380,	-0.882947592858927100,	-0.887010833178221820,	-0.891006524188367900,	-0.894934361602025000,	-0.898794046299166930,	
-0.902585284349860520,	-0.906307787036650050,	-0.909961270876543330,	-0.913545457642600980,	-0.917060074385124160,	-0.920504853452440370,	-0.923879532511286740,	-0.927183854566787420,	
-0.930417567982024570,	-0.933580426497201740,	-0.936672189248397570,	-0.939692620785908430,	-0.942641491092178320,	-0.945518575599316850,	-0.948323655206199430,	-0.951056516295153640,	
-0.953716950748226930,	-0.956304755963035550,	-0.958819734868193050,	-0.961261695938318890,	-0.963630453208622950,	-0.965925826289068310,	-0.968147640378107740,	-0.970295726275996470,	
-0.972369920397676670,	-0.974370064785235250,	-0.976296007119933360,	-0.978147600733805690,	-0.979924704620829590,	-0.981627183447663980,	-0.983254907563954620,	-0.984807753012208020,	
-0.986285601537231440,	-0.987688340595137660,	-0.989015863361916820,	-0.990268068741570360,	-0.991444861373810490,	-0.992546151641322090,	-0.993571855676587460,	-0.994521895368273400,	
-0.995396198367178850,	-0.996194698091745550,	-0.996917333733127960,	-0.997564050259824200,	-0.998134798421866920,	-0.998629534754573830,	-0.999048221581857800,	-0.999390827019095760,	
-0.999657324975557260,	-0.999847695156391270,	-0.999961923064171310,	-1.000000000000000000,	-0.999961923064171310,	-0.999847695156391270,	-0.999657324975557260,	-0.999390827019095760,	
-0.999048221581857800,	-0.998629534754573830,	-0.998134798421866920,	-0.997564050259824200,	-0.996917333733127960,	-0.996194698091745550,	-0.995396198367178850,	-0.994521895368273290,	
-0.993571855676587460,	-0.992546151641321980,	-0.991444861373810380,	-0.990268068741570250,	-0.989015863361916820,	-0.987688340595137770,	-0.986285601537231330,	-0.984807753012208020,	
-0.983254907563954620,	-0.981627183447663980,	-0.979924704620829590,	-0.978147600733805580,	-0.976296007119933360,	-0.974370064785235250,	-0.972369920397676560,	-0.970295726275996470,	
-0.968147640378107740,	-0.965925826289068310,	-0.963630453208622950,	-0.961261695938318890,	-0.958819734868193050,	-0.956304755963035440,	-0.953716950748226820,	-0.951056516295153530,	
-0.948323655206199320,	-0.945518575599316740,	-0.942641491092178320,	-0.939692620785908320,	-0.936672189248397570,	-0.933580426497201740,	-0.930417567982024570,	-0.927183854566787420,	
-0.923879532511286740,	-0.920504853452440260,	-0.917060074385124050,	-0.913545457642600870,	-0.909961270876543220,	-0.906307787036649940,	-0.902585284349860520,	-0.898794046299167040,	
-0.894934361602025110,	-0.891006524188367790,	-0.887010833178221600,	-0.882947592858926880,	-0.878817112661965380,	-0.874619707139395740,	-0.870355695939899700,	-0.866025403784438600,	
-0.861629160441525710,	-0.857167300702112220,	-0.852640164354092180,	-0.848048096156426070,	-0.843391445812885610,	-0.838670567945423940,	-0.833885822067168100,	-0.829037572555041740,	
-0.824126188622015700,	-0.819152044288991800,	-0.814115518356319120,	-0.809016994374947450,	-0.803856860617217280,	-0.798635510047292830,	-0.793353340291235170,	-0.788010753606722010,	
-0.782608156852413810,	-0.777145961456970790,	-0.771624583387720000,	-0.766044443118978010,	-0.760405965600031040,	-0.754709580222772010,	-0.748955720789002080,	-0.743144825477394240,	
-0.737277336810124080,	-0.731353701619170460,	-0.725374371012287520,	-0.719339800338651080,	-0.713250449154181560,	-0.707106781186547460,	-0.700909264299850900,	-0.694658370458997250,	
-0.688354575693753910,	-0.681998360062498480,	-0.675590207615660240,	-0.669130606358858240,	-0.662620048215737500,	-0.656059028990507160,	-0.649448048330183660,	-0.642787609686539250,	
-0.636078220277763950,	-0.629320391049837390,	-0.622514636637619520,	-0.615661475325658180,	-0.608761429008720660,	-0.601815023152048270,	-0.594822786751341260,	-0.587785252292473140,	
-0.580702955710939770,	-0.573576436351046050,	-0.566406236924832830,	-0.559192903470746900,	-0.551936985312058150,	-0.544639035015027080,	-0.537299608346823890,	-0.529919264233204900,	
-0.522498564715948800,	-0.515038074910054160,	-0.507538362960704090,	-0.499999999999999940,	-0.492423560103467050,	-0.484809620246337060,	-0.477158760259608410,	-0.469471562785890810,	
-0.461748613235033910,	-0.453990499739546750,	-0.446197813109808770,	-0.438371146789077400,	-0.430511096808295080,	-0.422618261740699440,	-0.414693242656239030,	-0.406736643075800150,	
-0.398749068925246200,	-0.390731128489273720,	-0.382683432365089780,	-0.374606593415912010,	-0.366501226724297250,	-0.358367949545300270,	-0.350207381259467380,	-0.342020143325668710,	
-0.333806859233770900,	-0.325568154457156640,	-0.317304656405092140,	-0.309016994374947400,	-0.300705799504273120,	-0.292371704722736770,	-0.284015344703922650,	-0.275637355816999160,	
-0.267238376078256850,	-0.258819045102520740,	-0.250380004054441440,	-0.241921895599667730,	-0.233445363855905390,	-0.224951054343865000,	-0.216439613938102880,	-0.207911690817759310,	
-0.199367934417197160,	-0.190808995376544800,	-0.182235525492147470,	-0.173648177666930330,	-0.165047605860677630,	-0.156434465040230870,	-0.147809411129610630,	-0.139173100960065440,	
-0.130526192220051570,	-0.121869343405147480,	-0.113203213767906720,	-0.104528463267653460,	-0.095845752520223981,	-0.087155742747658166,	-0.078459095727844944,	-0.069756473744125302,	
-0.061048539534856873,	-0.052335956242943828,	-0.043619387365336000,	-0.034899496702500969,	-0.026176948307873149,	-0.017452406437283512,	-0.008726535498373935,	0.000000000000000000,	
0.008726535498373935,	0.017452406437283512,	0.026176948307873149,	0.034899496702500969,	0.043619387365336000,	0.052335956242943828,	0.061048539534856873,	0.069756473744125302,	
0.078459095727844944,	0.087155742747658166,	0.095845752520223981,	0.104528463267653460,	0.113203213767906720,	0.121869343405147480,	0.130526192220051570,	0.139173100960065440,	
0.147809411129610630,	0.156434465040230870,	0.165047605860677630,	0.173648177666930330,	0.182235525492147470,	0.190808995376544800,	0.199367934417197160,	0.207911690817759310,	
0.216439613938102880,	0.224951054343865000,	0.233445363855905390,	0.241921895599667730,	0.250380004054441440,	0.258819045102520740,	0.267238376078256850,	0.275637355816999160,	
0.284015344703922650,	0.292371704722736770,	0.300705799504273120,	0.309016994374947400,	0.317304656405092140,	0.325568154457156640,	0.333806859233770900,	0.342020143325668710,	
0.350207381259467380,	0.358367949545300270,	0.366501226724297250,	0.374606593415912010,	0.382683432365089780,	0.390731128489273720,	0.398749068925246200,	0.406736643075800150,	
0.414693242656239030,	0.422618261740699440,	0.430511096808295080,	0.438371146789077400,	0.446197813109808770,	0.453990499739546750,	0.461748613235033910,	0.469471562785890810,	
0.477158760259608410,	0.484809620246337060,	0.492423560103467050,	0.499999999999999940,	0.507538362960704090,	0.515038074910054160,	0.522498564715948800,	0.529919264233204900,	
0.537299608346823890,	0.544639035015027080,	0.551936985312058150,	0.559192903470746900,	0.566406236924832830,	0.573576436351046050,	0.580702955710939770,	0.587785252292473140,	
0.594822786751341260,	0.601815023152048270,	0.608761429008720660,	0.615661475325658180,	0.622514636637619520,	0.629320391049837390,	0.636078220277763950,	0.642787609686539250,	
0.649448048330183660,	0.656059028990507160,	0.662620048215737500,	0.669130606358858240,	0.675590207615660240,	0.681998360062498480,	0.688354575693753910,	0.694658370458997250,	
0.700909264299850900,	0.707106781186547460,	0.713250449154181560,	0.719339800338651080,	0.725374371012287520,	0.731353701619170460,	0.737277336810124080,	0.743144825477394240,	
0.748955720789002080,	0.754709580222772010,	0.760405965600031040,	0.766044443118978010,	0.771624583387720000,	0.777145961456970790,	0.782608156852413810,	0.788010753606722010,	
0.793353340291235170,	0.798635510047292830,	0.803856860617217280,	0.809016994374947450,	0.814115518356319120,	0.819152044288991800,	0.824126188622015700,	0.829037572555041740,	
0.833885822067168100,	0.838670567945423940,	0.843391445812885610,	0.848048096156426070,	0.852640164354092180,	0.857167300702112220,	0.861629160441525710,	0.866025403784438600,	
0.870355695939899700,	0.874619707139395740,	0.878817112661965380,	0.882947592858926880,	0.887010833178221600,	0.891006524188367790,	0.894934361602025110,	0.898794046299167040,	
0.902585284349860520,	0.906307787036649940,	0.909961270876543220,	0.913545457642600870,	0.917060074385124050,	0.920504853452440260,	0.923879532511286740,	0.927183854566787420,	
0.930417567982024570,	0.933580426497201740,	0.936672189248397570,	0.939692620785908320,	0.942641491092178320,	0.945518575599316740,	0.948323655206199320,	0.951056516295153530,	
0.953716950748226820,	0.956304755963035440,	0.958819734868193050,	0.961261695938318890,	0.963630453208622950,	0.965925826289068310,	0.968147640378107740,	0.970295726275996470,	
0.972369920397676560,	0.974370064785235250,	0.976296007119933360,	0.978147600733805580,	0.979924704620829590,	0.981627183447663980,	0.983254907563954620,	0.984807753012208020,	
0.986285601537231330,	0.987688340595137770,	0.989015863361916820,	0.990268068741570250,	0.991444861373810380,	0.992546151641321980,	0.993571855676587460,	0.994521895368273290,	
0.995396198367178850,	0.996194698091745550,	0.996917333733127960,	0.997564050259824200,	0.998134798421866920,	0.998629534754573830,	0.999048221581857800,	0.999390827019095760,	
0.999657324975557260,	0.999847695156391270,	0.999961923064171310,	1.000000000000000000,	0.999961923064171310,	0.999847695156391270,	0.999657324975557260,	0.999390827019095760,	
0.999048221581857800,	0.998629534754573830,	0.998134798421866920,	0.997564050259824200,	0.996917333733127960,	0.996194698091745550,	0.995396198367178850,	0.994521895368273400,	
0.993571855676587460,	0.992546151641322090,	0.991444861373810490,	0.990268068741570360,	0.989015863361916820,	0.987688340595137660,	0.986285601537231440,	0.984807753012208020,	
0.983254907563954620,	0.981627183447663980,	0.979924704620829590,	0.978147600733805690,	0.976296007119933360,	0.974370064785235250,	0.972369920397676670,	0.970295726275996470,	
0.968147640378107740,	0.965925826289068310,	0.963630453208622950,	0.961261695938318890,	0.958819734868193050,	0.956304755963035550,	0.953716950748226930,	0.951056516295153640,	
0.948323655206199430,	0.945518575599316850,	0.942641491092178320,	0.939692620785908430,	0.936672189248397570,	0.933580426497201740,	0.930417567982024570,	0.927183854566787420,	
0.923879532511286740,	0.920504853452440370,	0.917060074385124160,	0.913545457642600980,	0.909961270876543330,	0.906307787036650050,	0.902585284349860520,	0.898794046299166930,	
0.894934361602025000,	0.891006524188367900,	0.887010833178221820,	0.882947592858927100,	0.878817112661965380,	0.874619707139395850,	0.870355695939899700,	0.866025403784438710,	
0.861629160441525710,	0.857167300702112330,	0.852640164354092290,	0.848048096156426070,	0.843391445812885830,	0.838670567945424050,	0.833885822067168210,	0.829037572555041740,	
0.824126188622015810,	0.819152044288992020,	0.814115518356319230,	0.809016994374947450,	0.803856860617217390,	0.798635510047292720,	0.793353340291235170,	0.788010753606722010,	
0.782608156852414030,	0.777145961456971010,	0.771624583387720220,	0.766044443118978010,	0.760405965600031040,	0.754709580222771790,	0.748955720789002080,	0.743144825477394240,	
0.737277336810124080,	0.731353701619170570,	0.725374371012287740,	0.719339800338651410,	0.713250449154181560,	0.707106781186547570,	0.700909264299851010,	0.694658370458997140,	
0.688354575693753910,	0.681998360062498590,	0.675590207615660350,	0.669130606358858350,	0.662620048215737390,	0.656059028990507280,	0.649448048330183770,	0.642787609686539470,	
0.636078220277764170,	0.629320391049837720,	0.622514636637619630,	0.615661475325658400,	0.608761429008720430,	0.601815023152048160,	0.594822786751341260,	0.587785252292473250,	
0.580702955710939880,	0.573576436351046380,	0.566406236924833160,	0.559192903470746900,	0.551936985312058260,	0.544639035015026860,	0.537299608346823780,	0.529919264233204900,	
0.522498564715948910,	0.515038074910054380,	0.507538362960704430,	0.499999999999999940,	0.492423560103467160,	0.484809620246337170,	0.477158760259608630,	0.469471562785891080,	
0.461748613235033910,	0.453990499739546860,	0.446197813109808990,	0.438371146789077290,	0.430511096808295080,	0.422618261740699500,	0.414693242656239190,	0.406736643075800430,	
0.398749068925246530,	0.390731128489274160,	0.382683432365089890,	0.374606593415912240,	0.366501226724297140,	0.358367949545300210,	0.350207381259467540,	0.342020143325668880,	
0.333806859233771180,	0.325568154457156980,	0.317304656405092140,	0.309016994374947510,	0.300705799504273340,	0.292371704722737050,	0.284015344703922590,	0.275637355816999660,	
0.267238376078257010,	0.258819045102521020,	0.250380004054441330,	0.241921895599667730,	0.233445363855905530,	0.224951054343864780,	0.216439613938103180,	0.207911690817759310,	
0.199367934417197690,	0.190808995376544970,	0.182235525492147720,	0.173648177666930280,	0.165047605860678100,	0.156434465040230980,	0.147809411129610380,	0.139173100960065740,	
0.130526192220051570,	0.121869343405147550,	0.113203213767906890,	0.104528463267653730,	0.095845752520223912,	0.087155742747658638,	0.078459095727845068,	0.069756473744125524,	
0.061048539534857192,	0.052335956242943807,	0.043619387365336069,	0.034899496702500699,	0.026176948307873423,	0.017452406437283439,	0.008726535498374403,	0.000000000000000122,	


};

float32_t arm_linear_interep_Costable[720] =  {

  -0.999961923064171310,	-0.999847695156391270,	-0.999657324975557260,	-0.999390827019095760,	-0.999048221581857800,	-0.998629534754573830,	-0.998134798421866920,	-0.997564050259824200,	
-0.996917333733127960,	-0.996194698091745550,	-0.995396198367178850,	-0.994521895368273290,	-0.993571855676587460,	-0.992546151641321980,	-0.991444861373810380,	-0.990268068741570250,	
-0.989015863361916820,	-0.987688340595137660,	-0.986285601537231330,	-0.984807753012208020,	-0.983254907563954510,	-0.981627183447663980,	-0.979924704620829480,	-0.978147600733805690,	
-0.976296007119933250,	-0.974370064785235250,	-0.972369920397676560,	-0.970295726275996470,	-0.968147640378107740,	-0.965925826289068200,	-0.963630453208622950,	-0.961261695938318670,	
-0.958819734868193050,	-0.956304755963035440,	-0.953716950748226820,	-0.951056516295153530,	-0.948323655206199320,	-0.945518575599316740,	-0.942641491092178320,	-0.939692620785908320,	
-0.936672189248397570,	-0.933580426497201740,	-0.930417567982024570,	-0.927183854566787310,	-0.923879532511286740,	-0.920504853452440150,	-0.917060074385123940,	-0.913545457642600760,	
-0.909961270876543100,	-0.906307787036649940,	-0.902585284349860630,	-0.898794046299167040,	-0.894934361602025000,	-0.891006524188367790,	-0.887010833178221710,	-0.882947592858926770,	
-0.878817112661965270,	-0.874619707139395740,	-0.870355695939899700,	-0.866025403784438710,	-0.861629160441525600,	-0.857167300702112220,	-0.852640164354092180,	-0.848048096156425960,	
-0.843391445812885720,	-0.838670567945424160,	-0.833885822067168100,	-0.829037572555041620,	-0.824126188622015480,	-0.819152044288991580,	-0.814115518356319120,	-0.809016994374947340,	
-0.803856860617217280,	-0.798635510047292940,	-0.793353340291235280,	-0.788010753606721900,	-0.782608156852413920,	-0.777145961456970680,	-0.771624583387719780,	-0.766044443118977900,	
-0.760405965600030820,	-0.754709580222772010,	-0.748955720789002190,	-0.743144825477394130,	-0.737277336810123970,	-0.731353701619170460,	-0.725374371012287630,	-0.719339800338651300,	
-0.713250449154181450,	-0.707106781186547460,	-0.700909264299850900,	-0.694658370458997030,	-0.688354575693753800,	-0.681998360062498370,	-0.675590207615660240,	-0.669130606358858240,	
-0.662620048215737610,	-0.656059028990507500,	-0.649448048330183550,	-0.642787609686539360,	-0.636078220277763730,	-0.629320391049837280,	-0.622514636637619410,	-0.615661475325658290,	
-0.608761429008720660,	-0.601815023152048380,	-0.594822786751341150,	-0.587785252292473030,	-0.580702955710939770,	-0.573576436351045830,	-0.566406236924832610,	-0.559192903470746680,	
-0.551936985312058150,	-0.544639035015027080,	-0.537299608346823550,	-0.529919264233204790,	-0.522498564715948800,	-0.515038074910054270,	-0.507538362960704310,	-0.499999999999999780,	
-0.492423560103467000,	-0.484809620246337000,	-0.477158760259608460,	-0.469471562785890530,	-0.461748613235033800,	-0.453990499739546750,	-0.446197813109808830,	-0.438371146789077510,	
-0.430511096808295360,	-0.422618261740699330,	-0.414693242656238800,	-0.406736643075800100,	-0.398749068925245980,	-0.390731128489273600,	-0.382683432365089730,	-0.374606593415912070,	
-0.366501226724297190,	-0.358367949545300270,	-0.350207381259467540,	-0.342020143325668710,	-0.333806859233771010,	-0.325568154457156420,	-0.317304656405091980,	-0.309016994374947340,	
-0.300705799504272950,	-0.292371704722736660,	-0.284015344703922650,	-0.275637355816999050,	-0.267238376078256850,	-0.258819045102520850,	-0.250380004054441390,	-0.241921895599667790,	
-0.233445363855905140,	-0.224951054343864810,	-0.216439613938102790,	-0.207911690817759120,	-0.199367934417197080,	-0.190808995376544800,	-0.182235525492147530,	-0.173648177666930300,	
-0.165047605860677710,	-0.156434465040231040,	-0.147809411129610440,	-0.139173100960065350,	-0.130526192220051380,	-0.121869343405147370,	-0.113203213767906710,	-0.104528463267653330,	
-0.095845752520223954,	-0.087155742747658235,	-0.078459095727844874,	-0.069756473744125330,	-0.061048539534857005,	-0.052335956242943620,	-0.043619387365335889,	-0.034899496702500733,	
-0.026176948307873017,	-0.017452406437283477,	-0.008726535498373775,	0.000000000000000061,	0.008726535498373897,	0.017452406437283376,	0.026176948307873139,	0.034899496702501080,	
0.043619387365336229,	0.052335956242943966,	0.061048539534856908,	0.069756473744125455,	0.078459095727844999,	0.087155742747658138,	0.095845752520224078,	0.104528463267653460,	
0.113203213767906610,	0.121869343405147490,	0.130526192220051710,	0.139173100960065690,	0.147809411129610770,	0.156434465040230920,	0.165047605860677820,	0.173648177666930410,	
0.182235525492147440,	0.190808995376544920,	0.199367934417197410,	0.207911690817759450,	0.216439613938102900,	0.224951054343864920,	0.233445363855905470,	0.241921895599667900,	
0.250380004054441500,	0.258819045102520740,	0.267238376078256960,	0.275637355816999160,	0.284015344703922530,	0.292371704722736770,	0.300705799504273290,	0.309016994374947450,	
0.317304656405092090,	0.325568154457156760,	0.333806859233771120,	0.342020143325668820,	0.350207381259467430,	0.358367949545300380,	0.366501226724297300,	0.374606593415911960,	
0.382683432365089840,	0.390731128489273940,	0.398749068925246250,	0.406736643075800210,	0.414693242656238910,	0.422618261740699440,	0.430511096808295250,	0.438371146789077460,	
0.446197813109808720,	0.453990499739546860,	0.461748613235034080,	0.469471562785890860,	0.477158760259608410,	0.484809620246337110,	0.492423560103467110,	0.500000000000000110,	
0.507538362960704200,	0.515038074910054380,	0.522498564715948910,	0.529919264233204900,	0.537299608346823890,	0.544639035015027200,	0.551936985312058260,	0.559192903470746790,	
0.566406236924832830,	0.573576436351046050,	0.580702955710939880,	0.587785252292473140,	0.594822786751341370,	0.601815023152048270,	0.608761429008720660,	0.615661475325658290,	
0.622514636637619630,	0.629320391049837500,	0.636078220277763950,	0.642787609686539360,	0.649448048330183550,	0.656059028990507280,	0.662620048215737500,	0.669130606358858240,	
0.675590207615660240,	0.681998360062498480,	0.688354575693754020,	0.694658370458997370,	0.700909264299850900,	0.707106781186547570,	0.713250449154181560,	0.719339800338651190,	
0.725374371012287740,	0.731353701619170570,	0.737277336810124080,	0.743144825477394240,	0.748955720789002080,	0.754709580222772010,	0.760405965600030930,	0.766044443118978010,	
0.771624583387720000,	0.777145961456970900,	0.782608156852413920,	0.788010753606722010,	0.793353340291235170,	0.798635510047292830,	0.803856860617217390,	0.809016994374947450,	
0.814115518356319230,	0.819152044288991800,	0.824126188622015700,	0.829037572555041620,	0.833885822067168210,	0.838670567945424050,	0.843391445812885720,	0.848048096156425960,	
0.852640164354092290,	0.857167300702112330,	0.861629160441525820,	0.866025403784438710,	0.870355695939899700,	0.874619707139395740,	0.878817112661965380,	0.882947592858926990,	
0.887010833178221710,	0.891006524188367900,	0.894934361602025110,	0.898794046299167040,	0.902585284349860630,	0.906307787036649940,	0.909961270876543220,	0.913545457642600870,	
0.917060074385124050,	0.920504853452440370,	0.923879532511286740,	0.927183854566787420,	0.930417567982024570,	0.933580426497201740,	0.936672189248397680,	0.939692620785908430,	
0.942641491092178430,	0.945518575599316850,	0.948323655206199320,	0.951056516295153530,	0.953716950748226930,	0.956304755963035440,	0.958819734868193050,	0.961261695938318890,	
0.963630453208622950,	0.965925826289068310,	0.968147640378107740,	0.970295726275996470,	0.972369920397676560,	0.974370064785235250,	0.976296007119933360,	0.978147600733805690,	
0.979924704620829590,	0.981627183447663980,	0.983254907563954620,	0.984807753012208020,	0.986285601537231440,	0.987688340595137770,	0.989015863361916820,	0.990268068741570360,	
0.991444861373810380,	0.992546151641321980,	0.993571855676587460,	0.994521895368273290,	0.995396198367178850,	0.996194698091745550,	0.996917333733127960,	0.997564050259824200,	
0.998134798421866920,	0.998629534754573830,	0.999048221581857800,	0.999390827019095760,	0.999657324975557260,	0.999847695156391270,	0.999961923064171310,	1.000000000000000000,	
0.999961923064171310,	0.999847695156391270,	0.999657324975557260,	0.999390827019095760,	0.999048221581857800,	0.998629534754573830,	0.998134798421866920,	0.997564050259824200,	
0.996917333733127960,	0.996194698091745550,	0.995396198367178850,	0.994521895368273290,	0.993571855676587460,	0.992546151641321980,	0.991444861373810380,	0.990268068741570360,	
0.989015863361916820,	0.987688340595137770,	0.986285601537231440,	0.984807753012208020,	0.983254907563954620,	0.981627183447663980,	0.979924704620829590,	0.978147600733805690,	
0.976296007119933360,	0.974370064785235250,	0.972369920397676560,	0.970295726275996470,	0.968147640378107740,	0.965925826289068310,	0.963630453208622950,	0.961261695938318890,	
0.958819734868193050,	0.956304755963035440,	0.953716950748226930,	0.951056516295153530,	0.948323655206199320,	0.945518575599316850,	0.942641491092178430,	0.939692620785908430,	
0.936672189248397680,	0.933580426497201740,	0.930417567982024570,	0.927183854566787420,	0.923879532511286740,	0.920504853452440370,	0.917060074385124050,	0.913545457642600870,	
0.909961270876543220,	0.906307787036649940,	0.902585284349860630,	0.898794046299167040,	0.894934361602025110,	0.891006524188367900,	0.887010833178221710,	0.882947592858926990,	
0.878817112661965380,	0.874619707139395740,	0.870355695939899700,	0.866025403784438710,	0.861629160441525820,	0.857167300702112330,	0.852640164354092290,	0.848048096156425960,	
0.843391445812885720,	0.838670567945424050,	0.833885822067168210,	0.829037572555041620,	0.824126188622015700,	0.819152044288991800,	0.814115518356319230,	0.809016994374947450,	
0.803856860617217390,	0.798635510047292830,	0.793353340291235170,	0.788010753606722010,	0.782608156852413920,	0.777145961456970900,	0.771624583387720000,	0.766044443118978010,	
0.760405965600030930,	0.754709580222772010,	0.748955720789002080,	0.743144825477394240,	0.737277336810124080,	0.731353701619170570,	0.725374371012287740,	0.719339800338651190,	
0.713250449154181560,	0.707106781186547570,	0.700909264299850900,	0.694658370458997370,	0.688354575693754020,	0.681998360062498480,	0.675590207615660240,	0.669130606358858240,	
0.662620048215737500,	0.656059028990507280,	0.649448048330183550,	0.642787609686539360,	0.636078220277763950,	0.629320391049837500,	0.622514636637619630,	0.615661475325658290,	
0.608761429008720660,	0.601815023152048270,	0.594822786751341370,	0.587785252292473140,	0.580702955710939880,	0.573576436351046050,	0.566406236924832830,	0.559192903470746790,	
0.551936985312058260,	0.544639035015027200,	0.537299608346823890,	0.529919264233204900,	0.522498564715948910,	0.515038074910054380,	0.507538362960704200,	0.500000000000000110,	
0.492423560103467110,	0.484809620246337110,	0.477158760259608410,	0.469471562785890860,	0.461748613235034080,	0.453990499739546860,	0.446197813109808720,	0.438371146789077460,	
0.430511096808295250,	0.422618261740699440,	0.414693242656238910,	0.406736643075800210,	0.398749068925246250,	0.390731128489273940,	0.382683432365089840,	0.374606593415911960,	
0.366501226724297300,	0.358367949545300380,	0.350207381259467430,	0.342020143325668820,	0.333806859233771120,	0.325568154457156760,	0.317304656405092090,	0.309016994374947450,	
0.300705799504273290,	0.292371704722736770,	0.284015344703922530,	0.275637355816999160,	0.267238376078256960,	0.258819045102520740,	0.250380004054441500,	0.241921895599667900,	
0.233445363855905470,	0.224951054343864920,	0.216439613938102900,	0.207911690817759450,	0.199367934417197410,	0.190808995376544920,	0.182235525492147440,	0.173648177666930410,	
0.165047605860677820,	0.156434465040230920,	0.147809411129610770,	0.139173100960065690,	0.130526192220051710,	0.121869343405147490,	0.113203213767906610,	0.104528463267653460,	
0.095845752520224078,	0.087155742747658138,	0.078459095727844999,	0.069756473744125455,	0.061048539534856908,	0.052335956242943966,	0.043619387365336229,	0.034899496702501080,	
0.026176948307873139,	0.017452406437283376,	0.008726535498373897,	0.000000000000000061,	-0.008726535498373775,	-0.017452406437283477,	-0.026176948307873017,	-0.034899496702500733,	
-0.043619387365335889,	-0.052335956242943620,	-0.061048539534857005,	-0.069756473744125330,	-0.078459095727844874,	-0.087155742747658235,	-0.095845752520223954,	-0.104528463267653330,	
-0.113203213767906710,	-0.121869343405147370,	-0.130526192220051380,	-0.139173100960065350,	-0.147809411129610440,	-0.156434465040231040,	-0.165047605860677710,	-0.173648177666930300,	
-0.182235525492147530,	-0.190808995376544800,	-0.199367934417197080,	-0.207911690817759120,	-0.216439613938102790,	-0.224951054343864810,	-0.233445363855905140,	-0.241921895599667790,	
-0.250380004054441390,	-0.258819045102520850,	-0.267238376078256850,	-0.275637355816999050,	-0.284015344703922650,	-0.292371704722736660,	-0.300705799504272950,	-0.309016994374947340,	
-0.317304656405091980,	-0.325568154457156420,	-0.333806859233771010,	-0.342020143325668710,	-0.350207381259467540,	-0.358367949545300270,	-0.366501226724297190,	-0.374606593415912070,	
-0.382683432365089730,	-0.390731128489273600,	-0.398749068925245980,	-0.406736643075800100,	-0.414693242656238800,	-0.422618261740699330,	-0.430511096808295360,	-0.438371146789077510,	
-0.446197813109808830,	-0.453990499739546750,	-0.461748613235033800,	-0.469471562785890530,	-0.477158760259608460,	-0.484809620246337000,	-0.492423560103467000,	-0.499999999999999780,	
-0.507538362960704310,	-0.515038074910054270,	-0.522498564715948800,	-0.529919264233204790,	-0.537299608346823550,	-0.544639035015027080,	-0.551936985312058150,	-0.559192903470746680,	
-0.566406236924832610,	-0.573576436351045830,	-0.580702955710939770,	-0.587785252292473030,	-0.594822786751341150,	-0.601815023152048380,	-0.608761429008720660,	-0.615661475325658290,	
-0.622514636637619410,	-0.629320391049837280,	-0.636078220277763730,	-0.642787609686539360,	-0.649448048330183550,	-0.656059028990507500,	-0.662620048215737610,	-0.669130606358858240,	
-0.675590207615660240,	-0.681998360062498370,	-0.688354575693753800,	-0.694658370458997030,	-0.700909264299850900,	-0.707106781186547460,	-0.713250449154181450,	-0.719339800338651300,	
-0.725374371012287630,	-0.731353701619170460,	-0.737277336810123970,	-0.743144825477394130,	-0.748955720789002190,	-0.754709580222772010,	-0.760405965600030820,	-0.766044443118977900,	
-0.771624583387719780,	-0.777145961456970680,	-0.782608156852413920,	-0.788010753606721900,	-0.793353340291235280,	-0.798635510047292940,	-0.803856860617217280,	-0.809016994374947340,	
-0.814115518356319120,	-0.819152044288991580,	-0.824126188622015480,	-0.829037572555041620,	-0.833885822067168100,	-0.838670567945424160,	-0.843391445812885720,	-0.848048096156425960,	
-0.852640164354092180,	-0.857167300702112220,	-0.861629160441525600,	-0.866025403784438710,	-0.870355695939899700,	-0.874619707139395740,	-0.878817112661965270,	-0.882947592858926770,	
-0.887010833178221710,	-0.891006524188367790,	-0.894934361602025000,	-0.898794046299167040,	-0.902585284349860630,	-0.906307787036649940,	-0.909961270876543100,	-0.913545457642600760,	
-0.917060074385123940,	-0.920504853452440150,	-0.923879532511286740,	-0.927183854566787310,	-0.930417567982024570,	-0.933580426497201740,	-0.936672189248397570,	-0.939692620785908320,	
-0.942641491092178320,	-0.945518575599316740,	-0.948323655206199320,	-0.951056516295153530,	-0.953716950748226820,	-0.956304755963035440,	-0.958819734868193050,	-0.961261695938318670,	
-0.963630453208622950,	-0.965925826289068200,	-0.968147640378107740,	-0.970295726275996470,	-0.972369920397676560,	-0.974370064785235250,	-0.976296007119933250,	-0.978147600733805690,	
-0.979924704620829480,	-0.981627183447663980,	-0.983254907563954510,	-0.984807753012208020,	-0.986285601537231330,	-0.987688340595137660,	-0.989015863361916820,	-0.990268068741570250,	
-0.991444861373810380,	-0.992546151641321980,	-0.993571855676587460,	-0.994521895368273290,	-0.995396198367178850,	-0.996194698091745550,	-0.996917333733127960,	-0.997564050259824200,	
-0.998134798421866920,	-0.998629534754573830,	-0.999048221581857800,	-0.999390827019095760,	-0.999657324975557260,	-0.999847695156391270,	-0.999961923064171310,	-1.000000000000000000,	

};


/* ----------------------------------------------------------------------
* Test input data for F32 SIN function
* Generated by the MATLAB rand() function
* --------------------------------------------------------------------*/
float32_t testInputSinCos_f32[TEST_LENGTH_SAMPLES] = 
{   
	0.649716504673081170,	2.501723745497831200,	0.188250329003310100,	0.432092748487532540,	1.722010988459680800,	1.788766476323060600,	1.786136060975809500,	0.056525543169408797,	
	0.491596272728153760,	0.262309671126153390
}; 

/*------------------------------------------------------------------------------
*  Reference out of SIN F32 function for Block Size = 32 
*  Caluclated from sin(in)
*------------------------------------------------------------------------------*/
float32_t testRefSinOutput32_f32[TEST_LENGTH_SAMPLES] =  
{ 
	0.604960695383043530,	0.597090287967934840,	0.187140422442966500,	0.418772124875992690,	0.988588831792106880,	0.976338412038794010,	0.976903856413481100,	0.056495446835214236,	
	0.472033731854734240,	0.259311907228582830
};

/*------------------------------------------------------------------------------
*  Method 1: Test out Buffer Caluclated from Cubic Interpolation
*------------------------------------------------------------------------------*/
float32_t testOutput[2 * TEST_LENGTH_SAMPLES];

/*------------------------------------------------------------------------------
*  Method 2: Test out Caluclated from Cubic Interpolation and Linear interpolation
*------------------------------------------------------------------------------*/
float32_t testLinIntOutput[2*TEST_LENGTH_SAMPLES];

/* ----------------------------------------------------------------------
* Global Variables for caluclating SNR's for Method1 & Method 2
* ------------------------------------------------------------------- */
float32_t snr1;
float32_t snr2;

float32_t sinVal, cosVal, sinValTest;

uint32_t blockSize = 100;

/* ----------------------------------------------------------------------------
* Caluclation of Sine values from Cubic Interpolation and Linear interpolation
* ---------------------------------------------------------------------------- */
int32_t main(void)
{
	uint32_t i, temp;
	arm_status status;
	float32_t xSpacing = 3.141592653589793e-004;
	float32_t x0, x, y0, y1;
	float32_t firstVal = -3.14159265358979;
	float32_t fract, theta;
	uint32_t index;


	/* Static Initialisations of Structere */
  	arm_linear_interp_instance_f32 S_Sin = {&arm_linear_interep_Sintable[0]}; 
	arm_linear_interp_instance_f32 S_Cos = {&arm_linear_interep_Costable[0]}; 
	
	x0 = -179.5;
	xSpacing = 0.5;

	/*------------------------------------------------------------------------------
	*  Method 1: Test out Caluclated from Linear Interpolation
	*------------------------------------------------------------------------------*/
	for(i=0; i< blockSize; i++)
  	{
      arm_sin_cos_f32(testInput_f32[i], &sinVal, &cosVal);
	  testOutput[2 * i] = sinVal;     
	  testOutput[2 * i + 1] = cosVal;
  	}
	/*------------------------------------------------------------------------------
	*  Method 2: Test out Caluclated from Linear interpolation
	*------------------------------------------------------------------------------*/
	for(i=0; i< TEST_LENGTH_SAMPLES; i++)
	{
		  index = (uint32_t)((testInput_f32[i] - x0) / xSpacing);

			theta = testInput_f32[i];
		  	if(theta > 0)
			{
				fract = testInput_f32[i] - (x0 + index * xSpacing);
			}
			else
			{
				fract = - theta + (x0 + index * xSpacing);
				fract = fract * 2;
			}

		  //fract =  testInput_f32[i] - (x0 + index * xSpacing);
	      testLinIntOutput[2 * i] = arm_linear_interp_f32(&S_Sin, fract, index);   
		  testLinIntOutput[2 * i + 1] = arm_linear_interp_f32(&S_Cos, fract, index);   
	}

//	for(i=0; i< TEST_LENGTH_SAMPLES; i++)
//	{
//		testOutput[i] = arm_sin_f32(testInputSinCos_f32[i]);
//	}
	
	/*------------------------------------------------------------------------------
	*  Method 2: Test out Caluclated from Cubic Interpolation and Linear interpolation
	*------------------------------------------------------------------------------*/
//	for(i=0; i< TEST_LENGTH_SAMPLES; i++)
//	{
//		x = testInputSinCos_f32[i];
//		temp = x/xSpacing;
//		
//		x0 = temp * xSpacing;
//		y0 =  arm_sin_f32(temp * xSpacing);
//		y1 =  arm_sin_f32((temp + 1) * xSpacing);
//
//		sinVal = arm_sin_f32(testInputSinCos_f32[i]);
////		
////		testLinIntOutput[i] = y0 + (x - x0) * (y1-y0)/xSpacing;	
//
//		
//
//		index = (uint32_t)((testInputSinCos_f32[i] - firstVal) / xSpacing);
//		//index += 1;
////		sinVal =   arm_linear_interep_table[index+1];
////		sinVal =   arm_linear_interep_table[index+2];
//	  	fract =  testInputSinCos_f32[i] - (firstVal + (index) * xSpacing);
//		//sinValTest = arm_linear_interp_f32(&S, fract, index);
//      	testLinIntOutput[i] = arm_linear_interp_f32(&S, fract, index);	
//	}
	
	/*------------------------------------------------------------------------------
	*  					SNR Caluclation for method 1
	*------------------------------------------------------------------------------*/  
	snr1 = arm_snr_f32(testRefOutput32_f32, testOutput, 2*TEST_LENGTH_SAMPLES);

	/*------------------------------------------------------------------------------
	*  					SNR Caluclation for method 2
	*------------------------------------------------------------------------------*/  
	snr2 = arm_snr_f32(testRefOutput32_f32, testLinIntOutput, 2*1);
	
	/*------------------------------------------------------------------------------
	*  					Initialise status depending on SNR caluclations
	*------------------------------------------------------------------------------*/ 
	if( snr2 > snr1)
	{
		status = ARM_SUCCESS;
	}
	else
	{
		status = ARM_TEST_FAILURE;
	}
	
	/* ----------------------------------------------------------------------
	** Loop here if the signals fail the PASS check.
	** This denotes a test failure
	** ------------------------------------------------------------------- */
	if( status != ARM_SUCCESS)
	{
		while(1);
	}
}

 /** \endlink */

